<?php
/**
 * MAXEDR10 - MAX ERP 10
 * Model: Conta (Contas a Pagar/Receber)
 */

namespace Maxedr10\Models;

use Maxedr10\Database\Database;

class Conta {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Lista contas com filtros
     */
    public function listar($empresaId, $filtros = []) {
        $sql = "SELECT c.*, cl.nome as cliente_nome
                FROM contas c
                LEFT JOIN clientes cl ON c.cliente_id = cl.id
                WHERE c.empresa_id = ?";
        
        $params = [$empresaId];
        
        // Filtro por tipo
        if (!empty($filtros['tipo'])) {
            $sql .= " AND c.tipo = ?";
            $params[] = $filtros['tipo'];
        }
        
        // Filtro por status
        if (!empty($filtros['status'])) {
            $sql .= " AND c.status = ?";
            $params[] = $filtros['status'];
        }
        
        // Filtro por data de vencimento
        if (!empty($filtros['data_vencimento_inicio'])) {
            $sql .= " AND c.data_vencimento >= ?";
            $params[] = $filtros['data_vencimento_inicio'];
        }
        
        if (!empty($filtros['data_vencimento_fim'])) {
            $sql .= " AND c.data_vencimento <= ?";
            $params[] = $filtros['data_vencimento_fim'];
        }
        
        $sql .= " ORDER BY c.data_vencimento ASC";
        
        return $this->db->query($sql, $params);
    }
    
    /**
     * Busca uma conta por ID
     */
    public function buscarPorId($id, $empresaId) {
        $sql = "SELECT * FROM contas WHERE id = ? AND empresa_id = ?";
        return $this->db->queryOne($sql, [$id, $empresaId]);
    }
    
    /**
     * Cria uma nova conta
     */
    public function criar($dados) {
        $sql = "INSERT INTO contas (
                    empresa_id, cliente_id, tipo, descricao, valor,
                    data_vencimento, data_pagamento, status, observacoes
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $dados['empresa_id'],
            $dados['cliente_id'] ?? null,
            $dados['tipo'],
            $dados['descricao'],
            $dados['valor'],
            $dados['data_vencimento'],
            $dados['data_pagamento'] ?? null,
            $dados['status'] ?? 'pendente',
            $dados['observacoes'] ?? null
        ];
        
        if ($this->db->execute($sql, $params)) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Atualiza uma conta
     */
    public function atualizar($id, $empresaId, $dados) {
        $sql = "UPDATE contas SET
                    cliente_id = ?, tipo = ?, descricao = ?, valor = ?,
                    data_vencimento = ?, data_pagamento = ?, status = ?, observacoes = ?
                WHERE id = ? AND empresa_id = ?";
        
        $params = [
            $dados['cliente_id'] ?? null,
            $dados['tipo'],
            $dados['descricao'],
            $dados['valor'],
            $dados['data_vencimento'],
            $dados['data_pagamento'] ?? null,
            $dados['status'] ?? 'pendente',
            $dados['observacoes'] ?? null,
            $id,
            $empresaId
        ];
        
        return $this->db->execute($sql, $params);
    }
    
    /**
     * Deleta uma conta
     */
    public function deletar($id, $empresaId) {
        $sql = "DELETE FROM contas WHERE id = ? AND empresa_id = ?";
        return $this->db->execute($sql, [$id, $empresaId]);
    }
    
    /**
     * Atualiza status de contas atrasadas
     */
    public function atualizarAtrasadas($empresaId) {
        $sql = "UPDATE contas 
                SET status = 'atrasado' 
                WHERE empresa_id = ? 
                    AND status = 'pendente' 
                    AND data_vencimento < CURDATE()";
        
        return $this->db->execute($sql, [$empresaId]);
    }
}
