<?php
/**
 * MAXEDR10 - MAX ERP 10
 * Model: Contrato
 */

namespace Maxedr10\Models;

use Maxedr10\Database\Database;

class Contrato {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    // ========== MODELOS DE CONTRATO ==========
    
    /**
     * Lista modelos de contrato
     */
    public function listarModelos($empresaId) {
        $sql = "SELECT * FROM contratos_modelos 
                WHERE empresa_id = ? AND ativo = 1 
                ORDER BY nome";
        
        return $this->db->query($sql, [$empresaId]);
    }
    
    /**
     * Busca um modelo por ID
     */
    public function buscarModeloPorId($id, $empresaId) {
        $sql = "SELECT * FROM contratos_modelos WHERE id = ? AND empresa_id = ?";
        return $this->db->queryOne($sql, [$id, $empresaId]);
    }
    
    /**
     * Cria um novo modelo de contrato
     */
    public function criarModelo($dados) {
        $sql = "INSERT INTO contratos_modelos (
                    empresa_id, nome, descricao, conteudo_html, tags
                ) VALUES (?, ?, ?, ?, ?)";
        
        $params = [
            $dados['empresa_id'],
            $dados['nome'],
            $dados['descricao'] ?? null,
            $dados['conteudo_html'],
            $dados['tags'] ?? null
        ];
        
        if ($this->db->execute($sql, $params)) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Atualiza um modelo de contrato
     */
    public function atualizarModelo($id, $empresaId, $dados) {
        $sql = "UPDATE contratos_modelos SET
                    nome = ?, descricao = ?, conteudo_html = ?, tags = ?
                WHERE id = ? AND empresa_id = ?";
        
        $params = [
            $dados['nome'],
            $dados['descricao'] ?? null,
            $dados['conteudo_html'],
            $dados['tags'] ?? null,
            $id,
            $empresaId
        ];
        
        return $this->db->execute($sql, $params);
    }
    
    /**
     * Deleta um modelo de contrato (soft delete)
     */
    public function deletarModelo($id, $empresaId) {
        $sql = "UPDATE contratos_modelos SET ativo = 0 WHERE id = ? AND empresa_id = ?";
        return $this->db->execute($sql, [$id, $empresaId]);
    }
    
    // ========== CONTRATOS GERADOS ==========
    
    /**
     * Lista contratos gerados
     */
    public function listarGerados($empresaId, $clienteId = null) {
        $sql = "SELECT cg.*, cl.nome as cliente_nome, cm.nome as modelo_nome
                FROM contratos_gerados cg
                INNER JOIN clientes cl ON cg.cliente_id = cl.id
                LEFT JOIN contratos_modelos cm ON cg.modelo_id = cm.id
                WHERE cg.empresa_id = ?";
        
        $params = [$empresaId];
        
        if ($clienteId) {
            $sql .= " AND cg.cliente_id = ?";
            $params[] = $clienteId;
        }
        
        $sql .= " ORDER BY cg.created_at DESC";
        
        return $this->db->query($sql, $params);
    }
    
    /**
     * Busca um contrato gerado por ID
     */
    public function buscarGeradoPorId($id, $empresaId) {
        $sql = "SELECT cg.*, cl.nome as cliente_nome, cm.nome as modelo_nome
                FROM contratos_gerados cg
                INNER JOIN clientes cl ON cg.cliente_id = cl.id
                LEFT JOIN contratos_modelos cm ON cg.modelo_id = cm.id
                WHERE cg.id = ? AND cg.empresa_id = ?";
        
        return $this->db->queryOne($sql, [$id, $empresaId]);
    }
    
    /**
     * Gera um novo contrato substituindo as tags
     */
    public function gerarContrato($empresaId, $clienteId, $modeloId, $dadosExtras = []) {
        // Buscar cliente
        $clienteModel = new Cliente();
        $cliente = $clienteModel->buscarPorId($clienteId, $empresaId);
        
        if (!$cliente) {
            return ['success' => false, 'message' => 'Cliente não encontrado'];
        }
        
        // Buscar modelo
        $modelo = $this->buscarModeloPorId($modeloId, $empresaId);
        
        if (!$modelo) {
            return ['success' => false, 'message' => 'Modelo de contrato não encontrado'];
        }
        
        // Preparar dados para substituição
        $substituicoes = [
            '{{nome}}' => $cliente['nome'] ?? '',
            '{{cpf}}' => $cliente['tipo'] === 'fisica' ? ($cliente['cpf_cnpj'] ?? '') : '',
            '{{cnpj}}' => $cliente['tipo'] === 'juridica' ? ($cliente['cpf_cnpj'] ?? '') : '',
            '{{cpf_cnpj}}' => $cliente['cpf_cnpj'] ?? '',
            '{{endereco}}' => $this->montarEnderecoCompleto($cliente),
            '{{cidade}}' => $cliente['cidade'] ?? '',
            '{{estado}}' => $cliente['estado'] ?? '',
            '{{telefone}}' => $cliente['telefone'] ?? '',
            '{{email}}' => $cliente['email'] ?? '',
            '{{data_atual}}' => date('d/m/Y'),
            '{{valor}}' => $dadosExtras['valor'] ?? '',
            '{{descricao_servico}}' => $dadosExtras['descricao_servico'] ?? '',
            '{{data_validade}}' => !empty($dadosExtras['data_validade']) ? date('d/m/Y', strtotime($dadosExtras['data_validade'])) : ''
        ];
        
        // Substituir tags no conteúdo
        $conteudoFinal = str_replace(
            array_keys($substituicoes),
            array_values($substituicoes),
            $modelo['conteudo_html']
        );
        
        // Salvar contrato gerado
        $sql = "INSERT INTO contratos_gerados (
                    empresa_id, cliente_id, modelo_id, conteudo_final, dados_extras
                ) VALUES (?, ?, ?, ?, ?)";
        
        $params = [
            $empresaId,
            $clienteId,
            $modeloId,
            $conteudoFinal,
            json_encode($dadosExtras)
        ];
        
        if ($this->db->execute($sql, $params)) {
            $contratoId = $this->db->lastInsertId();
            
            return [
                'success' => true,
                'contrato_id' => $contratoId,
                'conteudo_final' => $conteudoFinal,
                'pdf_path' => null // Simulação - aqui você poderia gerar um PDF real
            ];
        }
        
        return ['success' => false, 'message' => 'Erro ao gerar contrato'];
    }
    
    /**
     * Monta endereço completo do cliente
     */
    private function montarEnderecoCompleto($cliente) {
        $partes = array_filter([
            $cliente['endereco'] ?? '',
            $cliente['numero'] ?? '',
            $cliente['complemento'] ?? '',
            $cliente['bairro'] ?? ''
        ]);
        
        return implode(', ', $partes);
    }
}
