<?php
/**
 * MAXEDR10 - MAX ERP 10
 * Model: Lancamento
 */

namespace Maxedr10\Models;

use Maxedr10\Database\Database;

class Lancamento {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Lista lançamentos com filtros
     */
    public function listar($empresaId, $filtros = []) {
        $sql = "SELECT l.*, c.nome as categoria_nome, c.cor as categoria_cor,
                       cl.nome as cliente_nome
                FROM lancamentos_financeiros l
                LEFT JOIN categorias_financeiras c ON l.categoria_id = c.id
                LEFT JOIN clientes cl ON l.cliente_id = cl.id
                WHERE l.empresa_id = ?";
        
        $params = [$empresaId];
        
        // Filtro por data inicial
        if (!empty($filtros['data_inicial'])) {
            $sql .= " AND l.data >= ?";
            $params[] = $filtros['data_inicial'];
        }
        
        // Filtro por data final
        if (!empty($filtros['data_final'])) {
            $sql .= " AND l.data <= ?";
            $params[] = $filtros['data_final'];
        }
        
        // Filtro por tipo
        if (!empty($filtros['tipo'])) {
            $sql .= " AND l.tipo = ?";
            $params[] = $filtros['tipo'];
        }
        
        // Filtro por categoria
        if (!empty($filtros['categoria_id'])) {
            $sql .= " AND l.categoria_id = ?";
            $params[] = $filtros['categoria_id'];
        }
        
        $sql .= " ORDER BY l.data DESC, l.id DESC";
        
        return $this->db->query($sql, $params);
    }
    
    /**
     * Busca um lançamento por ID
     */
    public function buscarPorId($id, $empresaId) {
        $sql = "SELECT * FROM lancamentos_financeiros WHERE id = ? AND empresa_id = ?";
        return $this->db->queryOne($sql, [$id, $empresaId]);
    }
    
    /**
     * Cria um novo lançamento
     */
    public function criar($dados) {
        $sql = "INSERT INTO lancamentos_financeiros (
                    empresa_id, usuario_id, categoria_id, cliente_id, data, tipo,
                    descricao, valor, forma_pagamento, observacoes
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $dados['empresa_id'],
            $dados['usuario_id'],
            $dados['categoria_id'] ?? null,
            $dados['cliente_id'] ?? null,
            $dados['data'],
            $dados['tipo'],
            $dados['descricao'],
            $dados['valor'],
            $dados['forma_pagamento'] ?? 'dinheiro',
            $dados['observacoes'] ?? null
        ];
        
        if ($this->db->execute($sql, $params)) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Atualiza um lançamento
     */
    public function atualizar($id, $empresaId, $dados) {
        $sql = "UPDATE lancamentos_financeiros SET
                    categoria_id = ?, cliente_id = ?, data = ?, tipo = ?,
                    descricao = ?, valor = ?, forma_pagamento = ?, observacoes = ?
                WHERE id = ? AND empresa_id = ?";
        
        $params = [
            $dados['categoria_id'] ?? null,
            $dados['cliente_id'] ?? null,
            $dados['data'],
            $dados['tipo'],
            $dados['descricao'],
            $dados['valor'],
            $dados['forma_pagamento'] ?? 'dinheiro',
            $dados['observacoes'] ?? null,
            $id,
            $empresaId
        ];
        
        return $this->db->execute($sql, $params);
    }
    
    /**
     * Deleta um lançamento
     */
    public function deletar($id, $empresaId) {
        $sql = "DELETE FROM lancamentos_financeiros WHERE id = ? AND empresa_id = ?";
        return $this->db->execute($sql, [$id, $empresaId]);
    }
    
    /**
     * Retorna totais por tipo de lançamento (entrada/saída) em um período
     */
    public function totaisPorTipo($empresaId, $dataInicio, $dataFim) {
        $sql = "SELECT tipo, SUM(valor) as total
                FROM lancamentos_financeiros
                WHERE empresa_id = ? AND data BETWEEN ? AND ?
                GROUP BY tipo";
        
        $resultado = $this->db->query($sql, [$empresaId, $dataInicio, $dataFim]);
        
        $totais = [
            'entrada' => 0,
            'saida' => 0,
            'saldo' => 0
        ];
        
        foreach ($resultado as $row) {
            $totais[$row['tipo']] = (float) $row['total'];
        }
        
        $totais['saldo'] = $totais['entrada'] - $totais['saida'];
        
        return $totais;
    }
    
    /**
     * Retorna faturamento mensal dos últimos 12 meses
     */
    public function faturamentoMensal($empresaId, $meses = 12) {
        $sql = "SELECT 
                    DATE_FORMAT(data, '%Y-%m') as mes,
                    SUM(CASE WHEN tipo = 'entrada' THEN valor ELSE 0 END) as entradas,
                    SUM(CASE WHEN tipo = 'saida' THEN valor ELSE 0 END) as saidas
                FROM lancamentos_financeiros
                WHERE empresa_id = ? 
                    AND data >= DATE_SUB(CURDATE(), INTERVAL ? MONTH)
                GROUP BY mes
                ORDER BY mes";
        
        return $this->db->query($sql, [$empresaId, $meses]);
    }
}
